﻿using System;
using System.Linq;
using System.Reflection;
using System.IO;
using System.Configuration;
using System.ServiceProcess;
using System.Threading;
using log4net.Config;
using Spring.Context.Support; 
using gov.va.med.vbecs.Common.DependencyInjection;
using gov.va.med.vbecs.Common.Log;
using GlobalContext = log4net.GlobalContext;
using ILogger = gov.va.med.vbecs.Common.Log.ILogger;

namespace VBECS.Service
{
    static class Program
    {
        private static ILogger _logger;

        #region Constants

        public static readonly String Nl = Environment.NewLine;
        public static String ServiceAssemblyVersion = "NA"; 
        public static String ServiceCopyright = "NA"; 
        public static readonly String ServiceVersion = Assembly.GetExecutingAssembly().GetName().Version.ToString();

        #endregion

        #region Private

        static private void initialise_dependencies()
        {
            //Globally setup log output file name
            GlobalContext.Properties["LogName"] = Path.GetFileNameWithoutExtension(Assembly.GetEntryAssembly().Location);
            GlobalContext.Properties["LogNameDated"] = Path.GetFileNameWithoutExtension(Assembly.GetEntryAssembly().Location) + DateTime.Now.ToString("_MM-dd-yyyy");
            var log4NetFile = Path.Combine(Path.GetDirectoryName(typeof(Program).Assembly.Location) ?? ".", ConfigurationManager.AppSettings.Get("log4net.Config"));
            XmlConfigurator.ConfigureAndWatch(new FileInfo(log4NetFile));
            
            //Two configuration files are necessary since object in first file should be created and ready for object in the second one.
            var common = @"file://" + Path.Combine(Path.GetDirectoryName(typeof(Program).Assembly.Location) ?? ".", ConfigurationManager.AppSettings.Get("SpringConfigFileCommon"));
            var spring = @"file://" + Path.Combine(Path.GetDirectoryName(typeof(Program).Assembly.Location) ?? ".", ConfigurationManager.AppSettings.Get("SpringConfigFile"));
            DiContext.AppContext = new XmlApplicationContext( new[]{common, spring});

            //Here is the once-per-class call to initialize the log object
            //We have to do it after Spring.NET objects initialization.
            _logger = LogManager.Instance().LoggerLocator.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);
        }

        #endregion

        /// <summary>
        /// The main entry point for the application.
        /// </summary>
        static public int Main(string[] args)
        {
            var verAttr = Assembly.GetExecutingAssembly().GetCustomAttributes(typeof(AssemblyFileVersionAttribute), true).Cast<AssemblyFileVersionAttribute>().FirstOrDefault();
            if (verAttr != null)
                ServiceAssemblyVersion = verAttr.Version;
            var crightAttr = Assembly.GetExecutingAssembly().GetCustomAttributes(typeof(AssemblyCopyrightAttribute), true).Cast<AssemblyCopyrightAttribute>().FirstOrDefault();
            if (crightAttr != null)
                ServiceCopyright = crightAttr.Copyright;

            initialise_dependencies();

            // Started by user
            if (Environment.UserInteractive)
            {
                _logger.Debug("User interactive mode");
                // Show version information
                Console.WriteLine(Nl + Services.Common.AppServices.GlobalContext.Instance().ServiceProperties.Name + " " +
                    // If the version isn't the same as the file version, display both.
                    (ServiceVersion == ServiceAssemblyVersion ? ServiceVersion : ServiceVersion + " " + "[" + ServiceAssemblyVersion + "]") +
                    Nl + ServiceCopyright);

                // Parsing command line
                if (args != null && args.Length == 0)
                {
                    var sc = new ServiceController(Services.Common.AppServices.GlobalContext.Instance().ServiceProperties.Name, ".");
                    try
                    {
// ReSharper disable ReturnValueOfPureMethodIsNotUsed
                        sc.Status.ToString();  // throws exception if service does not exist
// ReSharper restore ReturnValueOfPureMethodIsNotUsed
                        Console.WriteLine(Nl + Nl + string.Format("Service [{0}] is already installed. Exit.", Services.Common.AppServices.GlobalContext.Instance().ServiceProperties.Name));
                        _logger.Info(string.Format("Service [{0}] is already installed. Exit.", Services.Common.AppServices.GlobalContext.Instance().ServiceProperties.Name));
                        return 0;
                    }
                    catch
                    {
                        Console.WriteLine(Environment.NewLine + Environment.NewLine +
                            "Install this service? Enter 'y' to confirm.");
                        var userInput = Console.ReadLine();
                        if (userInput == "y")
                            args = new[] { "-i" };
                        else
                        {
                            _logger.Debug(string.Format("User entered [{0}]. Exit.", userInput));
                            Console.WriteLine("Skip installation and exit");
                            return 0;
                        }
                    }
                }
                // Parsing command line
                if (args != null && args.Length >= 1)
                {
                    switch (args[0].ToLower())
                    {
                        case "-i":
                        case "/i":
                            SelfInstaller.InstallMe(Assembly.GetExecutingAssembly().Location, args);
                            return 0;
                        case "-u":
                        case "/u":
                            SelfInstaller.UninstallMe(Assembly.GetExecutingAssembly().Location);
                            return 0;
                        case "-c":
                        case "/c":
                            return LaunchConsole();
                    }
                }

                // Show usage
                Console.WriteLine(
                    Nl + Nl + "Usage: " + Path.GetFileName(Assembly.GetExecutingAssembly().Location) + " [/i | /u | /c]" +
                    Nl + Nl + "Where:" +
                    Nl + "\t/i - install service." +
                    Nl + "\t/u - uninstall service." +
                    Nl + "\t/c - run in console.");
            }
            else
            {
                _logger.Debug("Service mode");
                // Started by SCM
                // More than one user Service may run within the same process. To add
                // another service to this process, change the following line to
                // create a second service object. For example,
                var servicesToRun = new ServiceBase[] { new Service1() };
                ServiceBase.Run(servicesToRun);
            }

            return 0;
        }

        public static int LaunchConsole()
        {
            Console.WriteLine(Nl + Nl + "Starting... Press 'Q' or 'Esc' to exit");

            try
            {
                var server = Services.Common.AppServices.GlobalContext.Instance().Server;
                server.FatalErrorOccured += ServerOnFatalErrorOccured;
                try
                {
                    server.Start();
                    while (true)
                    {
                        Thread.Sleep(100);
                        var key = Console.ReadKey(true).Key;
                        if (key == ConsoleKey.Q) break;
                        if (key == ConsoleKey.Escape) break;
                    }
                    _logger.Debug(Nl + "Key was pressed. Start exiting..." + Nl);
                }
                finally
                {
                    server.Stop();   
                }

                return 0;
            }
            catch (Exception ex)
            {
                _logger.Error("LaunchConsole has failed", ex);
                Console.WriteLine(" --- ERROR --- " + Nl + "Check log file for more information");
                return -1;
            }

        }

        private static void ServerOnFatalErrorOccured(object sender, ThreadExceptionEventArgs threadExceptionEventArgs)
        {
            _logger.Error("Server process has failed", threadExceptionEventArgs.Exception);
            Console.WriteLine(" --- ERROR --- " + Nl + "Check log file for more information");
            Environment.Exit(-1);
        }
    }
}
